<?php
header('Content-Type: application/json');

// Get form data
$firstName = $_POST['firstName'] ?? '';
$lastName = $_POST['lastName'] ?? '';
$company = $_POST['company'] ?? '';
$relationStrength = $_POST['relationStrength'] ?? '';
$chemistry = $_POST['chemistry'] ?? '';
$email = $_POST['email'] ?? '';
$phone = $_POST['phone'] ?? '';
$comments = $_POST['comments'] ?? '';
$title = $_POST['title'] ?? '';
$departmentLeader = $_POST['departmentLeader'] ?? '';

// Validate required fields
if (empty($firstName) || empty($lastName) || empty($company) || empty($relationStrength) || empty($email)) {
    echo json_encode([
        'success' => false,
        'message' => 'Missing required fields'
    ]);
    exit;
}

// Validate email format
if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
    echo json_encode([
        'success' => false,
        'message' => 'Invalid email format'
    ]);
    exit;
}

try {
    // Step 1: Authenticate with WebCRM API
    $bearerToken = authenticateWebCRM();

    if (!$bearerToken) {
        throw new Exception('Failed to authenticate with WebCRM');
    }

    // Step 2: Get all organisations to find or create the specified one
    $organisations = getOrganisations($bearerToken);

    // Step 3: Check if organisation exists
    $orgId = null;
    $organisationExists = false;

    // Normalize input by trimming whitespace
    $companyNormalized = trim($company);

    foreach ($organisations as $org) {
        // Normalize organisation name from database
        $orgNameNormalized = trim($org['OrganisationName']);

        // Case-insensitive comparison
        if (strcasecmp($orgNameNormalized, $companyNormalized) === 0) {
            $orgId = $org['OrganisationId'];
            $organisationExists = true;

            // Log for debugging
            error_log("Found existing organisation: " . $org['OrganisationName'] . " (ID: $orgId)");
            break;
        }
    }

    // Step 4: Create organisation if it doesn't exist
    if (!$organisationExists) {
        error_log("Organisation '$companyNormalized' not found, creating new one");
        $orgId = createOrganisation($companyNormalized, $bearerToken);
        if (!$orgId) {
            throw new Exception('Failed to create organisation');
        }
        error_log("Created new organisation with ID: $orgId");
    }

    // Step 5: Get ALL persons in this organisation
    $existingPersons = getPersonsByOrganisation($orgId, $bearerToken);

    error_log("Found " . count($existingPersons) . " persons in organisation $orgId");

    $existingPerson = null;
    $normalizedEmail = strtolower(trim($email));

    // Search for matching person
    foreach ($existingPersons as $person) {
        if (!isset($person['PersonEmail'])) {
            continue;
        }

        $personEmail = strtolower(trim($person['PersonEmail']));

        error_log("Comparing: '$normalizedEmail' with '$personEmail'");

        if ($personEmail === $normalizedEmail) {
            $existingPerson = $person;
            error_log("MATCH FOUND: Person ID " . $person['PersonId'] . " with email $personEmail");
            break;
        }
    }

    // Step 6: Decide whether to create or update
    if ($existingPerson !== null) {
        // Person exists - check PersonCustom8 (PersonCustom8 is the department leader field in WebCRM)
        $personId = $existingPerson['PersonId'];
        $personCustom8 = $existingPerson['PersonCustom8'] ?? '';

        error_log("Person $personId has PersonCustom8: '$personCustom8'");

        if (empty(trim($personCustom8)) OR $personCustom8 == "NULL") {
            // PersonCustom8 is empty - UPDATE (PersonCustom8 is the department leader field in WebCRM)
            error_log("PersonCustom8 is empty, updating person $personId");

            $success = updatePerson(
                $personId,
                $firstName,
                $lastName,
                $phone,
                $email,
                $orgId,
                $relationStrength,
                $chemistry,
                $comments,
                $title,
                $departmentLeader,
                $bearerToken
            );

            if ($success) {
                echo json_encode([
                    'success' => true,
                    'message' => 'Person updated successfully',
                    'personId' => $personId,
                    'organisationId' => $orgId,
                    'action' => 'updated',
                    'organisationExisted' => $organisationExists
                ]);
            } else {
                throw new Exception('Failed to update person');
            }
        } else {
            // PersonCustom8 has a value - REJECT (PersonCustom8 is the department leader field in WebCRM)
            error_log("PersonCustom8 is not empty ('$personCustom8'), rejecting duplicate");

            echo json_encode([
                'success' => false,
                'message' => "En person med denne email eksisterer allerede og er tilknyttet $personCustom8"
            ]);
        }
        exit;
    }

    // Step 7: No existing person found - CREATE new one
    error_log("No existing person found with email $email, creating new person");
    $personId = createPerson($firstName, $lastName, $phone, $email, $orgId, $relationStrength, $chemistry, $comments, $title, $departmentLeader, $bearerToken);

    if ($personId && is_numeric($personId)) {
        echo json_encode([
            'success' => true,
            'message' => 'Person created successfully',
            'personId' => $personId,
            'organisationId' => $orgId,
            'action' => 'created',
            'organisationExisted' => $organisationExists
        ]);
    } else {
        // Check if it's a duplicate error from WebCRM
        if (is_array($personId) && isset($personId['Issues'])) {
            error_log("WebCRM returned duplicate error: " . json_encode($personId));

            // Try to get the existing person to retrieve PersonCustom8
            $existingPersons = getPersonsByOrganisation($orgId, $bearerToken);
            $departmentLeaderName = null;

            foreach ($existingPersons as $person) {
                if (!isset($person['PersonEmail'])) continue;
                $personEmail = strtolower(trim($person['PersonEmail']));

                if ($personEmail === $normalizedEmail) {
                    $departmentLeaderName = $person['PersonCustom8'] ?? null;
                    break;
                }
            }

            if (!empty(trim($departmentLeaderName))) {
                echo json_encode([
                    'success' => false,
                    'message' => "Personen eksisterer allerede og er tilknyttet $departmentLeaderName"
                ]);
            } else {
                echo json_encode([
                    'success' => false,
                    'message' => 'Personen eksisterer allerede men kunne ikke opdateres'
                ]);
            }
            exit;
        } else {
            throw new Exception('Failed to create person');
        }
    }


} catch (Exception $e) {
    error_log("Error in submit_person.php: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}

/**
 * Authenticate with WebCRM API
 */
function authenticateWebCRM() {
    $curl = curl_init();

    curl_setopt_array($curl, [
        CURLOPT_URL => "https://api.webcrm.com/Auth/apiLogin",
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_ENCODING => "",
        CURLOPT_MAXREDIRS => 10,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
        CURLOPT_CUSTOMREQUEST => "POST",
        CURLOPT_POSTFIELDS => "",
        CURLOPT_HTTPHEADER => [
            "Content-Type: application/json",
            "authCode: 34b18518-cfbb-4212-97ac-642dd3b8e4fb"
        ],
    ]);

    $response = curl_exec($curl);
    $err = curl_error($curl);
    curl_close($curl);

    if ($err) {
        error_log("cURL Error in authenticateWebCRM: " . $err);
        return false;
    }

    $data = json_decode($response, true);
    return $data['AccessToken'] ?? false;
}

/**
 * Get all organisations from WebCRM
 */
function getOrganisations($bearerToken) {
    $curl = curl_init();

    curl_setopt_array($curl, [
        CURLOPT_URL => "https://api.webcrm.com/Organisations?Size=1000",
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_ENCODING => "",
        CURLOPT_MAXREDIRS => 10,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
        CURLOPT_CUSTOMREQUEST => "GET",
        CURLOPT_POSTFIELDS => "",
        CURLOPT_HTTPHEADER => [
            "Authorization: Bearer $bearerToken",
        ],
    ]);

    $response = curl_exec($curl);
    $err = curl_error($curl);
    curl_close($curl);

    if ($err) {
        error_log("cURL Error in getOrganisations: " . $err);
        return [];
    }

    $responseData = json_decode($response, true) ?? [];

    // Log for debugging
    if (is_array($responseData)) {
        error_log("Retrieved " . count($responseData) . " organisations from WebCRM");
    }

    return $responseData;
}

/**
 * Create a new organisation in WebCRM
 */
function createOrganisation($organisationName, $bearerToken) {
    $curl = curl_init();

    $postData = json_encode([
        "OrganisationCompareName" => $organisationName,
        "OrganisationName" => $organisationName,
        "OrganisationId" => 0
    ]);

    curl_setopt_array($curl, [
        CURLOPT_URL => "https://api.webcrm.com/Organisations",
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_ENCODING => "",
        CURLOPT_MAXREDIRS => 10,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
        CURLOPT_CUSTOMREQUEST => "POST",
        CURLOPT_POSTFIELDS => $postData,
        CURLOPT_HTTPHEADER => [
            "Authorization: Bearer $bearerToken",
            "Content-Type: application/json",
        ],
    ]);

    $response = curl_exec($curl);
    $err = curl_error($curl);
    curl_close($curl);

    if ($err) {
        error_log("cURL Error in createOrganisation: " . $err);
        return false;
    }

    $data = json_decode($response, true);

    // Handle both numeric response and object response
    if (is_numeric($data)) {
        return (int)$data;
    } elseif (is_array($data) && isset($data['OrganisationId'])) {
        return (int)$data['OrganisationId'];
    }

    return $data['OrganisationId'] ?? $data ?? false;
}

/**
 * Get persons by organisation ID
 */
function getPersonsByOrganisation($orgId, $bearerToken) {
    $curl = curl_init();

    curl_setopt_array($curl, [
        CURLOPT_URL => "https://api.webcrm.com/Persons/ByOrganisation/$orgId?Size=1000",
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_ENCODING => "",
        CURLOPT_MAXREDIRS => 10,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
        CURLOPT_CUSTOMREQUEST => "GET",
        CURLOPT_POSTFIELDS => "",
        CURLOPT_HTTPHEADER => [
            "Authorization: Bearer $bearerToken",
        ],
    ]);

    $response = curl_exec($curl);
    $err = curl_error($curl);
    curl_close($curl);

    if ($err) {
        error_log("cURL Error in getPersonsByOrganisation: " . $err);
        return [];
    }

    $persons = json_decode($response, true) ?? [];

    error_log("getPersonsByOrganisation returned " . count($persons) . " persons for org $orgId");

    return $persons;
}

/**
 * Create a new person in WebCRM
 */
function createPerson($firstName, $lastName, $phone, $email, $orgId, $relationStrength, $chemistry, $comments, $title, $departmentLeader, $bearerToken) {
    $curl = curl_init();

    $postData = [
        "PersonDirectPhone" => $phone,
        "PersonEmail" => $email,
        "PersonFirstName" => $firstName,
        "PersonId" => 0,
        "PersonLastName" => $lastName,
        "PersonOrganisationId" => (int)$orgId,
        "PersonCustom7" => "Afdelingsleder Liste",
        "PersonCustom9" => $relationStrength,
        "PersonComment" => $comments,
        "PersonCustom8" => $departmentLeader,
        "PersonTitle" => $title,
    ];

    if (!empty($chemistry)) {
        $postData["PersonCustom10"] = $chemistry;
    }

    error_log("Creating person: " . $email . " in org " . $orgId);

    curl_setopt_array($curl, [
        CURLOPT_URL => "https://api.webcrm.com/Persons",
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_ENCODING => "",
        CURLOPT_MAXREDIRS => 10,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
        CURLOPT_CUSTOMREQUEST => "POST",
        CURLOPT_POSTFIELDS => json_encode($postData),
        CURLOPT_HTTPHEADER => [
            "Authorization: Bearer $bearerToken",
            "Content-Type: application/json",
        ],
    ]);

    $response = curl_exec($curl);
    $err = curl_error($curl);
    curl_close($curl);

    if ($err) {
        error_log("cURL Error in createPerson: " . $err);
        return false;
    }

    error_log("createPerson response: " . $response);

    $data = json_decode($response, true);

    // Handle both numeric response and object response
    if (is_numeric($data)) {
        return (int)$data;
    } elseif (is_array($data) && isset($data['PersonId']) && is_numeric($data['PersonId'])) {
        return (int)$data['PersonId'];
    }

    // Return the full response if it's an error (like duplicate)
    return $data;
}

/**
 * Update an existing person in WebCRM using PUT method
 */
function updatePerson($personId, $firstName, $lastName, $phone, $email, $orgId, $relationStrength, $chemistry, $comments, $title, $departmentLeader, $bearerToken) {
    $curl = curl_init();

    $postData = [
        "PersonDirectPhone" => $phone,
        "PersonEmail" => $email,
        "PersonFirstName" => $firstName,
        "PersonId" => (int)$personId,
        "PersonLastName" => $lastName,
        "PersonOrganisationId" => (int)$orgId,
        "PersonCustom7" => "Afdelingsleder Liste",
        "PersonCustom9" => $relationStrength,
        "PersonComment" => $comments,
        "PersonCustom8" => $departmentLeader,
        "PersonTitle" => $title,
    ];

    if (!empty($chemistry)) {
        $postData["PersonCustom10"] = $chemistry;
    }

    error_log("Updating person ID $personId with PersonCustom8: '$departmentLeader'");
    error_log("Update payload: " . json_encode($postData));

    curl_setopt_array($curl, [
        CURLOPT_URL => "https://api.webcrm.com/Persons/$personId",  // Include ID in URL
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_ENCODING => "",
        CURLOPT_MAXREDIRS => 10,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
        CURLOPT_CUSTOMREQUEST => "PUT",  // Changed from POST to PUT
        CURLOPT_POSTFIELDS => json_encode($postData),
        CURLOPT_HTTPHEADER => [
            "Authorization: Bearer $bearerToken",
            "Content-Type: application/json",
        ],
    ]);

    $response = curl_exec($curl);
    $err = curl_error($curl);
    curl_close($curl);

    if ($err) {
        error_log("cURL Error in updatePerson: " . $err);
        return false;
    }

    error_log("WebCRM updatePerson response: " . $response);

    $data = json_decode($response, true);

    // PUT request typically returns the updated person object or just success
    if (is_numeric($data) || (is_array($data) && isset($data['PersonId']))) {
        return true;
    }

    // Check for errors
    if (is_array($data) && isset($data['Issues'])) {
        error_log("Update failed with errors: " . json_encode($data));
        return false;
    }

    // If response is empty or just the person data, consider it success
    if ($response === '' || (is_array($data) && !empty($data))) {
        return true;
    }

    return false;
}
?>